#!/usr/bin/env python3
import os, subprocess, tempfile, time, re

AFTER_EFFECTS_APP = "Adobe After Effects 2025"

# --- PATHS ---
PROJECT   = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Baseball-Animation-1.aep"
CSV_PATH  = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/teams_rgb_correct.csv"
JSX_BATCH = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/batch_Baseball-RunScored-1.jsx"

# Folder that holds: 1.png,2.png,3.png,4.png, RUN SCORED.png, RUNS SCORED.png
TEXT_PNG_DIR = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/RunsScoredText"

# --- COMP / RENDER ---
COMP        = "RunScore-192"        # set to the comp you actually render
ANIM_NAME   = "RUNSCORE"
OUTDIR      = "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/RunScored_Animation_PNG_1_192"
PATH_TEMPLATE = "{league}"          # saves under .../{league}/<animName>/<frames>
OM_TEMPLATE   = "PNG Sequence"

RUN_TAG    = "R"
TAIL       = "_1_3"


RUN_MODE = "league"                 # or "subset"
LEAGUE   = "MLB"
SUBSET   = ["PHI","NYM","LAD"]      # test subset

def _write_bridge_jsx(env):
    def esc(s): return s.replace("\\","\\\\").replace('"','\\"')
    lines = [ '$.setenv("'+esc(k)+'","'+esc(str(v))+'");' for k,v in env.items() ]
    lines.append('$.evalFile(File("'+esc(JSX_BATCH)+'"));')
    code = "\n".join(lines)+"\n"
    tmp = tempfile.NamedTemporaryFile(prefix="ae_bridge_", suffix=".jsx", delete=False)
    tmp.write(code.encode("utf-8")); tmp.flush(); tmp.close()
    return tmp.name

def wait_for_ae_to_finish(directory, timeout=300):
    print("\nWaiting for After Effects to finish rendering...")
    start = time.time(); last = 0; stable = 0
    while time.time() - start < timeout:
        cnt = 0
        for root, _, files in os.walk(directory):
            for f in files:
                if re.match(r'.+\.png\d+$', f): cnt += 1
        if cnt > 0 and cnt == last:
            stable += 1
            if stable >= 3: return True
        else:
            stable = 0; last = cnt
        print(f"\rFiles found: {cnt} (waiting...)", end="", flush=True)
        time.sleep(1)
    return False

def fix_png_names(directory):
    print("\n\nFixing PNG file names...\n" + "-"*60)
    fixed = 0
    for root, _, files in os.walk(directory):
        for f in files:
            m = re.match(r'(.+)\.png(\d+)$', f)
            if not m: continue
            base, n = m.groups()
            oldp = os.path.join(root, f)
            newp = os.path.join(root, f"{base}_{n.zfill(5)}.png")
            try: os.rename(oldp, newp); fixed += 1; print(f"✓ {f} → {os.path.basename(newp)}")
            except Exception as e: print(f"✗ {f}: {e}")
    print(f"\nResults: Fixed {fixed} files")
    return fixed

def run(env):
    os.makedirs(OUTDIR, exist_ok=True)
    bridge = _write_bridge_jsx(env)
    osa = [
        "osascript",
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to activate',
        "-e", f'tell application "{AFTER_EFFECTS_APP}" to DoScriptFile POSIX file "{bridge}"',
    ]
    subprocess.Popen(osa)
    time.sleep(10)
    if wait_for_ae_to_finish(OUTDIR):
        time.sleep(5)
        fix_png_names(OUTDIR)
    print(f"\nOutput directory: {OUTDIR}")

def make_env(league=None, teams=None):
    env = {
        "AE_PROJECT": PROJECT,
        "AE_CSV": CSV_PATH,
        "AE_COMP": COMP,

        # Layers
        "AE_TEAMNAME_LAYER": "TeamName",
        "AE_TEAMLOGO_LAYER": "TeamLogo",
        "AE_EXPANDINGLINE_LAYER": "ExpandingLayer",

        # NEW: PNG replacement layers + directory
        "AE_RUNNUM_PNG_LAYER": "RunNumPng",
        "AE_SCORETEXT_PNG_LAYER": "RunScoredPng",
        "AE_TEXT_PNG_DIR": TEXT_PNG_DIR,
        
        # HomePlate still used (if present in your comp)
        "AE_HOMEPLATE_LAYER": "HomePlate-1.png",
        "AE_HOMEPLATE_PATH": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/BaseBall/HomeRunAnimation_1/HomePlate",

        # Logos
        "AE_LOGO_DIR": "/Users/zacharyladonis/Documents/GLANCE/Scoring Animations/Logos-Large/MLB",
        "AE_LOGO_EXTS": "png,jpg,jpeg,svg,ai,psd",

        # Output
        "AE_OUTDIR": OUTDIR,
        "AE_PATH_TEMPLATE": PATH_TEMPLATE,
        "AE_ANIM": ANIM_NAME,
        "AE_RS_TEMPLATE": "Best Settings",
        "AE_OM_TEMPLATE": OM_TEMPLATE,

        # Controls
        "AE_NO_RENDER": "0",
        "AE_PURGE_BEFORE_RENDER": "1",
        "AE_QUIT": "1",
        
        # File-name tags
        "AE_LEAGUE": league or LEAGUE,
        "AE_RUN_TAG": RUN_TAG,
        "AE_ANIM_TAIL": TAIL,
    }
    if teams:  env["AE_TEAMS"] = ",".join(teams)
    return env

if __name__ == "__main__":
    assert os.path.isfile(JSX_BATCH), f"JSX not found: {JSX_BATCH}"
    assert os.path.isfile(PROJECT),   f"Project not found: {PROJECT}"
    assert os.path.isfile(CSV_PATH),  f"CSV not found: {CSV_PATH}"
    if RUN_MODE == "league":
        run(make_env(league=LEAGUE))
    else:
        run(make_env(teams=SUBSET))
